<?php
/**
 * SERVER VERIFICATION SCRIPT
 * Upload this to your server and run it to verify all validation bypasses are in place
 */

echo "🔍 WHATMARK VALIDATION BYPASS VERIFICATION\n";
echo "==========================================\n\n";

$basePath = __DIR__;
$issuesFound = false;

function checkFile($path, $description) {
    if (!file_exists($path)) {
        echo "❌ $description: FILE NOT FOUND ($path)\n";
        return false;
    } else {
        echo "✅ $description: EXISTS\n";
        return true;
    }
}

function checkContent($path, $search, $description, $shouldExist = true) {
    if (!file_exists($path)) {
        echo "❌ $description: FILE NOT FOUND\n";
        return false;
    }
    
    $content = file_get_contents($path);
    $found = strpos($content, $search) !== false;
    
    if ($shouldExist && $found) {
        echo "✅ $description: FOUND\n";
        return true;
    } elseif (!$shouldExist && !$found) {
        echo "✅ $description: CORRECTLY ABSENT\n";
        return true;
    } elseif (!$shouldExist && $found) {
        echo "❌ $description: STILL PRESENT (should be removed)\n";
        return false;
    } else {
        echo "❌ $description: NOT FOUND\n";
        return false;
    }
}

function checkMiddlewareChain($path) {
    if (!file_exists($path)) return false;
    
    $content = file_get_contents($path);
    
    // Check that admin routes use only ['web'] middleware
    if (preg_match("/Route::middleware\(\s*\[\s*'web',\s*'validate'\s*\]\s*\)/", $content)) {
        echo "❌ Admin routes still have 'validate' middleware\n";
        return false;
    }
    
    if (preg_match("/Route::middleware\(\s*\[\s*'web'\s*\]\s*\)/", $content)) {
        echo "✅ Admin routes use clean ['web'] middleware\n";
        return true;
    }
    
    echo "⚠️  Cannot determine admin middleware configuration\n";
    return false;
}

echo "1. CHECKING CRITICAL FILES EXIST:\n";
echo "=================================\n";

$filesExist = [
    $basePath . '/bootstrap/app.php' => 'bootstrap/app.php',
    $basePath . '/routes/web.php' => 'routes/web.php',
    $basePath . '/app/Http/Middleware/CheckDatabaseVersion.php' => 'CheckDatabaseVersion middleware',
    $basePath . '/storage/.installed' => 'Installation marker',
];

foreach ($filesExist as $path => $desc) {
    if (!checkFile($path, $desc)) {
        $issuesFound = true;
    }
}

echo "\n2. CHECKING BOOTSTRAP CONFIGURATION:\n";
echo "===================================\n";

if (checkContent($basePath . '/bootstrap/app.php', "Route::middleware(['web'])", "Admin routes use ['web'] only")) {
    // Good
} else {
    $issuesFound = true;
}

if (checkContent($basePath . '/bootstrap/app.php', "'validate'", "No 'validate' middleware alias", false)) {
    // Good - validate not found
} else {
    echo "❌ 'validate' middleware alias still present in bootstrap/app.php\n";
    $issuesFound = true;
}

if (checkContent($basePath . '/bootstrap/app.php', "CheckDatabaseVersion", "No CheckDatabaseVersion middleware", false)) {
    echo "✅ CheckDatabaseVersion middleware correctly removed\n";
} else {
    echo "❌ CheckDatabaseVersion middleware still referenced\n";
    $issuesFound = true;
}

echo "\n3. CHECKING ROUTES:\n";
echo "==================\n";

if (checkContent($basePath . '/routes/web.php', '/validate', "No validation routes", false)) {
    echo "✅ No validation routes found\n";
} else {
    echo "❌ Validation routes still present\n";
    $issuesFound = true;
}

if (checkContent($basePath . '/routes/web.php', 'InstallController', "No InstallController references", false)) {
    echo "✅ InstallController references removed\n";
} else {
    echo "❌ InstallController still referenced\n";
    $issuesFound = true;
}

echo "\n4. CHECKING MIDDLEWARE:\n";
echo "======================\n";

if (checkContent($basePath . '/app/Http/Middleware/CheckDatabaseVersion.php', 'return \$next(\$request)', "CheckDatabaseVersion bypassed")) {
    // Good
} else {
    echo "❌ CheckDatabaseVersion middleware not properly bypassed\n";
    $issuesFound = true;
}

echo "\n5. CHECKING CACHE FILES:\n";
echo "=======================\n";

$cacheDir = $basePath . '/bootstrap/cache/';
if (is_dir($cacheDir)) {
    $cacheFiles = glob($cacheDir . '*.php');
    if (empty($cacheFiles)) {
        echo "✅ Bootstrap cache is empty\n";
    } else {
        echo "⚠️  Bootstrap cache files still exist:\n";
        foreach ($cacheFiles as $file) {
            echo "   - " . basename($file) . "\n";
        }
    }
} else {
    echo "✅ Bootstrap cache directory doesn't exist\n";
}

echo "\n6. RUNNING CACHE CLEAR COMMANDS:\n";
echo "=================================\n";

$commands = [
    'php artisan config:clear' => 'Config cache',
    'php artisan route:clear' => 'Route cache',
    'php artisan view:clear' => 'View cache',
    'php artisan cache:clear' => 'Application cache',
];

foreach ($commands as $cmd => $desc) {
    echo "Running: $cmd\n";
    exec($cmd . ' 2>&1', $output, $returnCode);
    if ($returnCode === 0) {
        echo "✅ $desc cleared successfully\n";
    } else {
        echo "⚠️  $desc clear command failed (may not be critical)\n";
        // Don't mark as issue since artisan might not be available
    }
}

echo "\n" . str_repeat("=", 50) . "\n";
if ($issuesFound) {
    echo "❌ ISSUES FOUND - Please fix the issues above before testing\n";
} else {
    echo "✅ ALL CHECKS PASSED!\n";
    echo "Your application should now work without validation errors.\n";
}
echo str_repeat("=", 50) . "\n";

echo "\n📋 NEXT STEPS:\n";
echo "==============\n";

if ($issuesFound) {
    echo "1. Fix all the issues marked with ❌ above\n";
    echo "2. Re-run this verification script\n";
    echo "3. If all checks pass, restart your web server:\n";
} else {
    echo "1. Restart your web server:\n";
    echo "   sudo systemctl restart apache2\n";
    echo "   OR\n";
    echo "   sudo systemctl restart nginx\n";
    echo "2. Visit your application URL\n";
}

echo "   sudo systemctl restart apache2  # or nginx\n";
echo "3. Visit your domain (https://projectnow.run.place)\n";
echo "4. You should see the login page, not the validation page\n\n";

echo "🔗 Direct URL to test:\n";
echo "https://projectnow.run.place/login\n";
echo "https://projectnow.run.place/admin\n\n";

echo "If you still get errors after this, please contact support with this verification output.\n";
?>
