<?php

namespace Corbital\Installer;

use Illuminate\Support\Facades\File;

class Installer
{
    /**
     * Check if the application is already installed.
     */
    public static function isAppInstalled(): bool
    {
        $installedFile = static::installedFileLocation();

        return (file_exists($installedFile)) ? static::verifyInstalledFile($installedFile) : false;
    }

    /**
     * Get the full path to the installed file.
     */
    public static function installedFileLocation(): string
    {
        return base_path(config('installer.storage_path', 'storage') . '/' . config('installer.installed_file', '.installed'));
    }

    /**
     * Check if the current request is for the installation wizard.
     */
    public static function isInstalling(): bool
    {
        return str_starts_with(request()->path(), config('installer.install_route', 'install'));
    }

    /**
     * Check if the application requires installation.
     */
    public static function requiresInstallation(): bool
    {
        return ! static::isAppInstalled();
    }

    /**
     * Mark the application as installed.
     */
    public function markAsInstalled(): bool
    {
        $installedFile = static::installedFileLocation();

        if (file_exists($installedFile)) {
            return static::verifyInstalledFile($installedFile);
        }

        // FIXED: Create simple content without database dependency
        $content = "WhatsMark Installation Completed\n" .
                  "Date: " . date('Y-m-d H:i:s') . "\n" .
                  "Version: 1.0.0\n" .
                  "Status: INSTALLED\n";

        // Ensure storage directory exists
        $storagePath = base_path(config('installer.storage_path', 'storage'));
        if (! File::exists($storagePath)) {
            File::makeDirectory($storagePath, 0755, true);
        }

        // Write the installed file
        $bytes = File::put($installedFile, $content);

        return $bytes !== false;
    }

    /**
     * Get the installation file content, if it exists.
     */
    public static function getInstallationInfo(): ?array
    {
        $file = static::installedFileLocation();

        if (! file_exists($file)) {
            return null;
        }

        $content = file_get_contents($file);
        $lines   = explode("\n", $content);
        $info    = [];

        foreach ($lines as $line) {
            if (strpos($line, ':') !== false) {
                [$key, $value]    = explode(':', $line, 2);
                $info[trim($key)] = trim($value);
            }
        }

        return $info;
    }

    /**
     * Remove the installed file (useful for resetting installation).
     */
    public static function reset(): bool
    {
        $file = static::installedFileLocation();

        if (file_exists($file)) {
            return unlink($file);
        }

        return true;
    }

    /**
     * SIMPLE FILE-ONLY VERIFICATION - No database checks!
     */
    public static function verifyInstalledFile($file): bool
    {
        // Simple checks only - no database dependency
        if (!file_exists($file)) {
            return false;
        }

        $file_content = file_get_contents($file);

        // Check if file has content
        if (empty($file_content)) {
            return false;
        }

        // Simple verification - if file exists and has content, consider it installed
        return true;
    }
}
