<?php
/**
 * COMPREHENSIVE CACHE CLEARING SCRIPT
 * This script will completely clear ALL Laravel caches to resolve the 'validate' middleware error
 */

echo "🧹 COMPREHENSIVE CACHE CLEARING SCRIPT\n";
echo "=====================================\n\n";

$basePath = __DIR__;

// Directories to clear
$cacheDirs = [
    $basePath . '/bootstrap/cache/',
    $basePath . '/storage/framework/cache/',
    $basePath . '/storage/framework/sessions/',
    $basePath . '/storage/framework/testing/',
    $basePath . '/storage/framework/views/',
];

$filesToDelete = [
    // Bootstrap cache files
    $basePath . '/bootstrap/cache/*.php',
    $basePath . '/bootstrap/cache/*.tmp',
    
    // Storage cache files
    $basePath . '/storage/framework/cache/data/*',
    
    // Laravel compiled files
    $basePath . '/storage/framework/*.php',
    $basePath . '/storage/*.php',
    
    // Compiled service container
    $basePath . '/bootstrap/cache/compiled.php',
    $basePath . '/bootstrap/cache/services.php',
    $basePath . '/bootstrap/cache/container.php',
];

function deleteDirectory($dir) {
    if (!file_exists($dir)) return;
    
    $files = array_diff(scandir($dir), array('.', '..'));
    foreach ($files as $file) {
        $filePath = $dir . '/' . $file;
        if (is_dir($filePath)) {
            deleteDirectory($filePath);
        } else {
            unlink($filePath);
            echo "  ✅ Deleted: $filePath\n";
        }
    }
}

function deleteFilesByPattern($pattern) {
    $files = glob($pattern);
    foreach ($files as $file) {
        if (is_file($file)) {
            unlink($file);
            echo "  ✅ Deleted: $file\n";
        }
    }
}

echo "1. Clearing bootstrap cache directory...\n";
deleteDirectory($basePath . '/bootstrap/cache/');

echo "\n2. Clearing storage framework cache...\n";
foreach ($cacheDirs as $dir) {
    if (file_exists($dir)) {
        echo "  Clearing: $dir\n";
        deleteDirectory($dir);
    }
}

echo "\n3. Deleting cached files by pattern...\n";
foreach ($filesToDelete as $pattern) {
    echo "  Pattern: $pattern\n";
    deleteFilesByPattern($pattern);
}

echo "\n4. Checking for any remaining cache files...\n";
$remainingFiles = [];
$searchPaths = [
    $basePath . '/bootstrap/cache/',
    $basePath . '/storage/framework/',
    $basePath . '/storage/',
];

foreach ($searchPaths as $path) {
    if (file_exists($path)) {
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            if ($file->getExtension() === 'php' || $file->getExtension() === 'tmp') {
                $remainingFiles[] = $file->getRealPath();
            }
        }
    }
}

if (empty($remainingFiles)) {
    echo "  ✅ No remaining cache files found!\n";
} else {
    echo "  ⚠️  Found remaining cache files:\n";
    foreach ($remainingFiles as $file) {
        echo "    - $file\n";
    }
}

echo "\n5. Running Laravel cache clear commands...\n";
$commands = [
    'php artisan config:clear',
    'php artisan route:clear',
    'php artisan view:clear',
    'php artisan cache:clear',
    'php artisan optimize:clear',
];

foreach ($commands as $command) {
    echo "  Running: $command\n";
    exec($command . ' 2>&1', $output, $returnCode);
    if ($returnCode === 0) {
        echo "    ✅ Success\n";
    } else {
        echo "    ⚠️  Command failed (this is normal if artisan isn't available)\n";
    }
}

echo "\n🎉 CACHE CLEARING COMPLETED!\n";
echo "==============================\n\n";

echo "NEXT STEPS:\n";
echo "1. Restart your web server:\n";
echo "   sudo systemctl restart apache2\n";
echo "   OR\n";
echo "   sudo systemctl restart nginx\n\n";

echo "2. Visit your application URL\n";
echo "   You should now see the login page instead of the 'validate' error\n\n";

echo "3. If you still get errors, check:\n";
echo "   - All 4 files were uploaded to FTP correctly\n";
echo "   - Web server was restarted\n";
echo "   - No validation routes exist in routes/web.php\n\n";

echo "✅ COMPLETE - The application should now work without validation errors!\n";
?>
